from mongoengine import Document, StringField, DateTimeField, BooleanField, IntField
from datetime import datetime, timedelta
import uuid


class OTP(Document):
    """OTP model for password reset and verification"""
    meta = {'collection': 'otps'}

    otp_id = StringField(required=True, unique=True, default=lambda: str(uuid.uuid4()))
    user_id = StringField(required=True)
    email = StringField(required=True)
    otp_code = StringField(required=True)
    otp_type = StringField(required=True, choices=['forgot_password', 'email_verification'])
    is_used = BooleanField(default=False)
    is_expired = BooleanField(default=False)
    attempts = IntField(default=0)
    max_attempts = IntField(default=5)
    created_at = DateTimeField(default=datetime.utcnow)
    expires_at = DateTimeField(required=True)
    used_at = DateTimeField()

    # Device and location info
    device_type = StringField()
    device_fingerprint = StringField()
    mac_id = StringField()
    ip_address = StringField()
    location = StringField()

    def is_valid(self):
        """Check if OTP is valid"""
        if self.is_used:
            return False
        if self.is_expired:
            return False
        if datetime.utcnow() > self.expires_at:
            self.is_expired = True
            self.save()
            return False
        if self.attempts >= self.max_attempts:
            self.is_expired = True
            self.save()
            return False
        return True

    def mark_as_used(self):
        """Mark OTP as used"""
        self.is_used = True
        self.used_at = datetime.utcnow()
        self.save()

    def increment_attempts(self):
        """Increment verification attempts"""
        self.attempts += 1
        if self.attempts >= self.max_attempts:
            self.is_expired = True
        self.save()

    @staticmethod
    def generate_otp_code(length=6):
        """Generate random OTP code"""
        import random
        import string
        return ''.join(random.choices(string.digits, k=length))

    def __repr__(self):
        return f'<OTP {self.email} - {self.otp_type}>'
