from flask import request, jsonify
from flask_jwt_extended import jwt_required, get_jwt_identity
from werkzeug.utils import secure_filename
from app.api.v1 import api_v1_bp
from app.models.user import User
from app.models.case import Case
from app.models.document import CaseDocument
from app.models.action_log import ActionLog
from app.utils.response_helper import success_response, error_response
from datetime import datetime
import uuid
import os


# ============================================================================
# USER PROFILE ENDPOINTS
# ============================================================================

@api_v1_bp.route('/users/get_user_profile', methods=['GET'])
@jwt_required()
def get_user_profile():
    """
    Get User Profile
    ---
    tags:
      - User Profile
    security:
      - Bearer: []
    parameters:
      - name: Authorization
        in: header
        type: string
        required: true
        description: Bearer token
      - name: body
        in: body
        required: false
        schema:
          type: object
          properties:
            device_type:
              type: string
            device_fingerprint:
              type: string
            mac_id:
              type: string
            location:
              type: object
    responses:
      200:
        description: Profile fetched successfully
      401:
        description: Unauthorized
    """
    request_id = str(uuid.uuid4())
    action_log_id = str(uuid.uuid4())

    try:
        current_user_id = get_jwt_identity()
        data = request.get_json() or {}

        # Get headers
        ip_address = request.headers.get('X-Real-IP', request.remote_addr)
        user_agent = request.headers.get('User-Agent', '')

        # Find user
        user = User.objects(user_id=current_user_id).first()

        if not user:
            ActionLog(
                action_log_id=action_log_id,
                action_type='get_profile',
                ip_address=ip_address,
                user_agent=user_agent,
                response_status='failed',
                error_details={'reason': 'User not found'}
            ).save()

            return error_response(
                message="User not found",
                error_code="USER_NOT_FOUND",
                error_details="User profile not found",
                status_code=404,
                request_id=request_id,
                action_log_id=action_log_id
            )

        # Check if user is active
        if not user.is_active:
            return error_response(
                message="Account deactivated",
                error_code="ACCOUNT_DEACTIVATED",
                error_details="Your account has been deactivated",
                status_code=403,
                request_id=request_id,
                action_log_id=action_log_id
            )

        # Log successful action
        ActionLog(
            action_log_id=action_log_id,
            user_id=user.user_id,
            action_type='get_profile',
            entity_type='user',
            entity_id=user.user_id,
            ip_address=ip_address,
            user_agent=user_agent,
            response_status='success'
        ).save()

        # Prepare response
        profile_data = {
            "user_id": user.user_id,
            "email": user.email,
            "full_name": user.full_name,
            "mobile": user.phone or "",
            "user_type": user.user_type,
            "created_at": user.created_at.isoformat() if user.created_at else None,
            "updated_at": user.updated_at.isoformat() if user.updated_at else None,
            "settings": user.profile_data or {}
        }

        return success_response(
            message="Profile fetched successfully",
            data=profile_data,
            status_code=200,
            request_id=request_id,
            action_log_id=action_log_id
        )

    except Exception as e:
        ActionLog(
            action_log_id=action_log_id,
            action_type='get_profile',
            ip_address=request.headers.get('X-Real-IP', request.remote_addr),
            user_agent=request.headers.get('User-Agent', ''),
            response_status='error',
            error_details={'exception': str(e)}
        ).save()

        return error_response(
            message="Internal server error",
            error_code="INTERNAL_ERROR",
            error_details=str(e),
            status_code=500,
            request_id=request_id,
            action_log_id=action_log_id
        )


@api_v1_bp.route('/users/update_user_profile', methods=['PUT'])
@jwt_required()
def update_user_profile():
    """
    Update User Profile
    ---
    tags:
      - User Profile
    security:
      - Bearer: []
    parameters:
      - name: Authorization
        in: header
        type: string
        required: true
      - name: body
        in: body
        required: true
        schema:
          type: object
          properties:
            full_name:
              type: string
            mobile:
              type: string
            device_type:
              type: string
            device_fingerprint:
              type: string
            mac_id:
              type: string
            location:
              type: object
    responses:
      200:
        description: Profile updated successfully
      400:
        description: Validation error
    """
    request_id = str(uuid.uuid4())
    action_log_id = str(uuid.uuid4())

    try:
        current_user_id = get_jwt_identity()
        data = request.get_json()

        # Get headers
        ip_address = request.headers.get('X-Real-IP', request.remote_addr)
        user_agent = request.headers.get('User-Agent', '')

        # Find user
        user = User.objects(user_id=current_user_id).first()

        if not user:
            return error_response(
                message="User not found",
                error_code="USER_NOT_FOUND",
                error_details="User profile not found",
                status_code=404,
                request_id=request_id,
                action_log_id=action_log_id
            )

        # Check if user is active
        if not user.is_active:
            return error_response(
                message="Account deactivated",
                error_code="ACCOUNT_DEACTIVATED",
                error_details="Your account has been deactivated",
                status_code=403,
                request_id=request_id,
                action_log_id=action_log_id
            )

        # Update fields
        updated_fields = []
        if 'full_name' in data and data['full_name']:
            user.full_name = data['full_name']
            updated_fields.append('full_name')

        if 'mobile' in data:
            user.phone = data['mobile']
            updated_fields.append('mobile')

        user.updated_at = datetime.utcnow()
        user.save()

        # Log successful action
        ActionLog(
            action_log_id=action_log_id,
            user_id=user.user_id,
            action_type='update_profile',
            entity_type='user',
            entity_id=user.user_id,
            ip_address=ip_address,
            user_agent=user_agent,
            request_data={"updated_fields": updated_fields},
            response_status='success'
        ).save()

        response_data = {
            "user_id": user.user_id,
            "full_name": user.full_name,
            "mobile": user.phone or "",
            "updated_at": user.updated_at.isoformat()
        }

        return success_response(
            message="Profile updated successfully",
            data=response_data,
            status_code=200,
            request_id=request_id,
            action_log_id=action_log_id
        )

    except Exception as e:
        return error_response(
            message="Internal server error",
            error_code="INTERNAL_ERROR",
            error_details=str(e),
            status_code=500,
            request_id=request_id,
            action_log_id=action_log_id
        )


# Due to length, I'll create this in multiple files. Let me continue with case management endpoints...

# ============================================================================
# CASE MANAGEMENT ENDPOINTS
# ============================================================================

@api_v1_bp.route('/user/get_all_cases', methods=['GET'])
@jwt_required()
def get_all_cases():
    """
    Get All Cases
    ---
    tags:
      - Case Management
    security:
      - Bearer: []
    parameters:
      - name: page
        in: query
        type: integer
        default: 1
      - name: limit
        in: query
        type: integer
        default: 10
      - name: search
        in: query
        type: string
    responses:
      200:
        description: Cases fetched successfully
    """
    request_id = str(uuid.uuid4())
    action_log_id = str(uuid.uuid4())

    try:
        current_user_id = get_jwt_identity()
        
        # Get query parameters
        page = int(request.args.get('page', 1))
        limit = int(request.args.get('limit', 10))
        search = request.args.get('search', '')

        # Get headers
        ip_address = request.headers.get('X-Real-IP', request.remote_addr)
        user_agent = request.headers.get('User-Agent', '')

        # Build query
        query = {'user_id': current_user_id, 'is_deleted': False}
        
        if search:
            query['$or'] = [
                {'name': {'$regex': search, '$options': 'i'}},
                {'client': {'$regex': search, '$options': 'i'}},
                {'case_number': {'$regex': search, '$options': 'i'}}
            ]

        # Get total count
        total_items = Case.objects(**query).count()
        total_pages = (total_items + limit - 1) // limit

        # Get paginated cases
        skip = (page - 1) * limit
        cases = Case.objects(**query).order_by('-created_at').skip(skip).limit(limit)

        # Format response
        cases_data = []
        for case in cases:
            cases_data.append({
                "case_id": case.case_id,
                "name": case.name,
                "client": case.client,
                "status": case.status,
                "case_number": case.case_number,
                "created_at": case.created_at.isoformat() if case.created_at else None,
                "start_date": case.start_date.isoformat() if case.start_date else None,
                "end_date": case.end_date.isoformat() if case.end_date else None,
                "documents_count": case.documents_count,
                "ai_exports": case.ai_exports_count
            })

        response_data = {
            "cases": cases_data,
            "pagination": {
                "page": page,
                "limit": limit,
                "total_pages": total_pages,
                "total_items": total_items
            }
        }

        # Log action
        ActionLog(
            action_log_id=action_log_id,
            user_id=current_user_id,
            action_type='get_all_cases',
            ip_address=ip_address,
            user_agent=user_agent,
            response_status='success'
        ).save()

        return success_response(
            message="Cases fetched successfully",
            data=response_data,
            status_code=200,
            request_id=request_id,
            action_log_id=action_log_id
        )

    except Exception as e:
        return error_response(
            message="Internal server error",
            error_code="INTERNAL_ERROR",
            error_details=str(e),
            status_code=500,
            request_id=request_id,
            action_log_id=action_log_id
        )


@api_v1_bp.route('/user/add_cases', methods=['POST'])
@jwt_required()
def add_case():
    """
    Add New Case
    ---
    tags:
      - Case Management
    security:
      - Bearer: []
    parameters:
      - name: body
        in: body
        required: true
        schema:
          type: object
          required:
            - name
            - client
          properties:
            name:
              type: string
            client:
              type: string
            description:
              type: string
            start_date:
              type: string
              format: date-time
            end_date:
              type: string
              format: date-time
    responses:
      201:
        description: Case created successfully
    """
    request_id = str(uuid.uuid4())
    action_log_id = str(uuid.uuid4())

    try:
        current_user_id = get_jwt_identity()
        data = request.get_json()

        # Validate required fields
        if not data.get('name'):
            return error_response(
                message="Validation error",
                error_code="VALIDATION_ERROR",
                error_details="Case name is required",
                status_code=400,
                request_id=request_id,
                action_log_id=action_log_id
            )

        if not data.get('client'):
            return error_response(
                message="Validation error",
                error_code="VALIDATION_ERROR",
                error_details="Client name is required",
                status_code=400,
                request_id=request_id,
                action_log_id=action_log_id
            )

        # Create case
        case = Case(
            name=data['name'],
            client=data['client'],
            description=data.get('description', ''),
            case_number=Case.generate_case_number(),
            user_id=current_user_id,
            created_by=current_user_id,
            status='active'
        )

        # Set dates if provided
        if data.get('start_date'):
            case.start_date = datetime.fromisoformat(data['start_date'].replace('Z', '+00:00'))
        if data.get('end_date'):
            case.end_date = datetime.fromisoformat(data['end_date'].replace('Z', '+00:00'))

        case.save()

        # Log action
        ActionLog(
            action_log_id=action_log_id,
            user_id=current_user_id,
            action_type='add_case',
            entity_type='case',
            entity_id=case.case_id,
            ip_address=request.headers.get('X-Real-IP', request.remote_addr),
            user_agent=request.headers.get('User-Agent', ''),
            request_data={"name": case.name, "client": case.client},
            response_status='success'
        ).save()

        response_data = {
            "case_id": case.case_id,
            "name": case.name,
            "client": case.client,
            "case_number": case.case_number,
            "status": case.status,
            "created_at": case.created_at.isoformat()
        }

        return success_response(
            message="Case created successfully",
            data=response_data,
            status_code=201,
            request_id=request_id,
            action_log_id=action_log_id
        )

    except Exception as e:
        return error_response(
            message="Internal server error",
            error_code="INTERNAL_ERROR",
            error_details=str(e),
            status_code=500,
            request_id=request_id,
            action_log_id=action_log_id
        )


@api_v1_bp.route('/user/get_case_by_id/<case_id>', methods=['GET'])
@jwt_required()
def get_case_by_id(case_id):
    """
    Get Case By ID
    ---
    tags:
      - Case Management
    security:
      - Bearer: []
    parameters:
      - name: case_id
        in: path
        type: string
        required: true
    responses:
      200:
        description: Case details fetched
      404:
        description: Case not found
    """
    request_id = str(uuid.uuid4())
    action_log_id = str(uuid.uuid4())

    try:
        current_user_id = get_jwt_identity()

        # Find case
        case = Case.objects(case_id=case_id, user_id=current_user_id, is_deleted=False).first()

        if not case:
            return error_response(
                message="Case not found",
                error_code="CASE_NOT_FOUND",
                error_details="No case found with this ID",
                status_code=404,
                request_id=request_id,
                action_log_id=action_log_id
            )

        response_data = {
            "case_id": case.case_id,
            "name": case.name,
            "client": case.client,
            "description": case.description or "",
            "status": case.status,
            "case_number": case.case_number,
            "created_at": case.created_at.isoformat() if case.created_at else None,
            "updated_at": case.updated_at.isoformat() if case.updated_at else None,
            "start_date": case.start_date.isoformat() if case.start_date else None,
            "end_date": case.end_date.isoformat() if case.end_date else None,
            "documents_count": case.documents_count
        }

        return success_response(
            message="Case details fetched",
            data=response_data,
            status_code=200,
            request_id=request_id,
            action_log_id=action_log_id
        )

    except Exception as e:
        return error_response(
            message="Internal server error",
            error_code="INTERNAL_ERROR",
            error_details=str(e),
            status_code=500,
            request_id=request_id,
            action_log_id=action_log_id
        )


@api_v1_bp.route('/user/update_case_by_id/<case_id>', methods=['PUT'])
@jwt_required()
def update_case_by_id(case_id):
    """
    Update Case By ID
    ---
    tags:
      - Case Management
    security:
      - Bearer: []
    parameters:
      - name: case_id
        in: path
        type: string
        required: true
      - name: body
        in: body
        schema:
          type: object
          properties:
            name:
              type: string
            status:
              type: string
            client:
              type: string
            description:
              type: string
    responses:
      200:
        description: Case updated successfully
    """
    request_id = str(uuid.uuid4())
    action_log_id = str(uuid.uuid4())

    try:
        current_user_id = get_jwt_identity()
        data = request.get_json()

        # Find case
        case = Case.objects(case_id=case_id, user_id=current_user_id, is_deleted=False).first()

        if not case:
            return error_response(
                message="Case not found",
                error_code="CASE_NOT_FOUND",
                error_details="No case found with this ID",
                status_code=404,
                request_id=request_id,
                action_log_id=action_log_id
            )

        # Update fields
        if 'name' in data:
            case.name = data['name']
        if 'status' in data:
            case.status = data['status']
        if 'client' in data:
            case.client = data['client']
        if 'description' in data:
            case.description = data['description']

        case.updated_by = current_user_id
        case.updated_at = datetime.utcnow()
        case.save()

        response_data = {
            "case_id": case.case_id,
            "name": case.name,
            "status": case.status,
            "updated_at": case.updated_at.isoformat()
        }

        return success_response(
            message="Case updated successfully",
            data=response_data,
            status_code=200,
            request_id=request_id,
            action_log_id=action_log_id
        )

    except Exception as e:
        return error_response(
            message="Internal server error",
            error_code="INTERNAL_ERROR",
            error_details=str(e),
            status_code=500,
            request_id=request_id,
            action_log_id=action_log_id
        )


@api_v1_bp.route('/user/delete_cases_by_id/<case_id>', methods=['DELETE'])
@jwt_required()
def delete_case_by_id(case_id):
    """
    Delete Case By ID
    ---
    tags:
      - Case Management
    security:
      - Bearer: []
    parameters:
      - name: case_id
        in: path
        type: string
        required: true
    responses:
      200:
        description: Case deleted successfully
    """
    request_id = str(uuid.uuid4())
    action_log_id = str(uuid.uuid4())

    try:
        current_user_id = get_jwt_identity()

        # Find case
        case = Case.objects(case_id=case_id, user_id=current_user_id, is_deleted=False).first()

        if not case:
            return error_response(
                message="Case not found",
                error_code="CASE_NOT_FOUND",
                error_details="No case found with this ID",
                status_code=404,
                request_id=request_id,
                action_log_id=action_log_id
            )

        # Soft delete
        case.soft_delete(current_user_id)

        response_data = {
            "case_id": case.case_id,
            "deleted_at": case.deleted_at.isoformat(),
            "can_be_restored": True
        }

        return success_response(
            message="Case deleted successfully",
            data=response_data,
            status_code=200,
            request_id=request_id,
            action_log_id=action_log_id
        )

    except Exception as e:
        return error_response(
            message="Internal server error",
            error_code="INTERNAL_ERROR",
            error_details=str(e),
            status_code=500,
            request_id=request_id,
            action_log_id=action_log_id
        )


# ============================================================================
# DOCUMENT MANAGEMENT ENDPOINTS  
# ============================================================================

@api_v1_bp.route('/documents/upload', methods=['POST'])
@jwt_required()
def upload_document():
    """
    Upload Document
    ---
    tags:
      - Document Management
    security:
      - Bearer: []
    consumes:
      - multipart/form-data
    parameters:
      - name: file
        in: formData
        type: file
        required: true
      - name: case_id
        in: formData
        type: string
        required: true
      - name: document_type
        in: formData
        type: string
        enum: [question, answer, evidence, contract, report, other]
    responses:
      201:
        description: Document uploaded successfully
    """
    request_id = str(uuid.uuid4())
    action_log_id = str(uuid.uuid4())

    try:
        current_user_id = get_jwt_identity()

        # Check if file is in request
        if 'file' not in request.files:
            return error_response(
                message="Upload failed",
                error_code="NO_FILE",
                error_details="No file provided",
                status_code=400,
                request_id=request_id,
                action_log_id=action_log_id
            )

        file = request.files['file']
        case_id = request.form.get('case_id')
        document_type = request.form.get('document_type', 'question')

        if not case_id:
            return error_response(
                message="Validation error",
                error_code="VALIDATION_ERROR",
                error_details="case_id is required",
                status_code=400,
                request_id=request_id,
                action_log_id=action_log_id
            )

        # Verify case exists and belongs to user
        case = Case.objects(case_id=case_id, user_id=current_user_id, is_deleted=False).first()
        if not case:
            return error_response(
                message="Case not found",
                error_code="CASE_NOT_FOUND",
                error_details="Case does not exist",
                status_code=404,
                request_id=request_id,
                action_log_id=action_log_id
            )

        # Get file info
        original_filename = secure_filename(file.filename)
        file_extension = CaseDocument.get_file_extension(original_filename)
        
        # Create unique filename
        unique_filename = f"{uuid.uuid4()}_{original_filename}"
        
        # Save file (simplified - in production, use cloud storage)
        upload_dir = os.path.join('uploads', case_id)
        os.makedirs(upload_dir, exist_ok=True)
        file_path = os.path.join(upload_dir, unique_filename)
        file.save(file_path)
        
        # Get file size
        file_size = os.path.getsize(file_path)
        
        # Check file size limit (50MB)
        if file_size > 50 * 1024 * 1024:
            os.remove(file_path)
            return error_response(
                message="Upload failed",
                error_code="FILE_TOO_LARGE",
                error_details="File size exceeds maximum limit of 50MB",
                status_code=400,
                request_id=request_id,
                action_log_id=action_log_id
            )

        # Create document record
        document = CaseDocument(
            case_id=case_id,
            user_id=current_user_id,
            file_name=unique_filename,
            original_file_name=original_filename,
            file_path=file_path,
            file_size=file_size,
            file_extension=file_extension,
            document_type=document_type,
            processing_status='queued'
        )
        document.save()

        # Update case document count
        case.documents_count += 1
        case.save()

        response_data = {
            "document_id": document.document_id,
            "file_name": original_filename,
            "file_size": file_size,
            "case_id": case_id,
            "document_type": document_type,
            "uploaded_at": document.uploaded_at.isoformat(),
            "processing_status": document.processing_status
        }

        return success_response(
            message="Document uploaded successfully",
            data=response_data,
            status_code=201,
            request_id=request_id,
            action_log_id=action_log_id
        )

    except Exception as e:
        return error_response(
            message="Internal server error",
            error_code="INTERNAL_ERROR",
            error_details=str(e),
            status_code=500,
            request_id=request_id,
            action_log_id=action_log_id
        )


@api_v1_bp.route('/get_all_documents_by_case_id', methods=['GET'])
@jwt_required()
def get_all_documents_by_case_id():
    """
    Get All Documents By Case ID
    ---
    tags:
      - Document Management
    security:
      - Bearer: []
    parameters:
      - name: case_id
        in: query
        type: string
        required: true
      - name: type
        in: query
        type: string
      - name: page
        in: query
        type: integer
        default: 1
    responses:
      200:
        description: Documents fetched successfully
    """
    request_id = str(uuid.uuid4())
    action_log_id = str(uuid.uuid4())

    try:
        current_user_id = get_jwt_identity()
        
        case_id = request.args.get('case_id')
        doc_type = request.args.get('type')
        page = int(request.args.get('page', 1))
        limit = int(request.args.get('limit', 20))

        if not case_id:
            return error_response(
                message="Validation error",
                error_code="VALIDATION_ERROR",
                error_details="case_id is required",
                status_code=400,
                request_id=request_id,
                action_log_id=action_log_id
            )

        # Verify case exists and belongs to user
        case = Case.objects(case_id=case_id, user_id=current_user_id, is_deleted=False).first()
        if not case:
            return error_response(
                message="Case not found",
                error_code="CASE_NOT_FOUND",
                error_details="Case does not exist",
                status_code=404,
                request_id=request_id,
                action_log_id=action_log_id
            )

        # Build query
        query = {'case_id': case_id, 'user_id': current_user_id, 'is_deleted': False}
        if doc_type:
            query['document_type'] = doc_type

        # Get total count
        total_items = CaseDocument.objects(**query).count()
        total_pages = (total_items + limit - 1) // limit

        # Get paginated documents
        skip = (page - 1) * limit
        documents = CaseDocument.objects(**query).order_by('-uploaded_at').skip(skip).limit(limit)

        # Format response
        documents_data = []
        for doc in documents:
            documents_data.append({
                "document_id": doc.document_id,
                "file_name": doc.original_file_name,
                "document_type": doc.document_type,
                "file_size": doc.file_size,
                "uploaded_at": doc.uploaded_at.isoformat(),
                "processing_status": doc.processing_status
            })

        response_data = {
            "documents": documents_data,
            "pagination": {
                "page": page,
                "limit": limit,
                "total_pages": total_pages,
                "total_items": total_items
            }
        }

        return success_response(
            message="Documents fetched successfully",
            data=response_data,
            status_code=200,
            request_id=request_id,
            action_log_id=action_log_id
        )

    except Exception as e:
        return error_response(
            message="Internal server error",
            error_code="INTERNAL_ERROR",
            error_details=str(e),
            status_code=500,
            request_id=request_id,
            action_log_id=action_log_id
        )


@api_v1_bp.route('/get_documents_by_id/<document_id>', methods=['GET'])
@jwt_required()
def get_document_by_id(document_id):
    """
    Get Document By ID
    ---
    tags:
      - Document Management
    security:
      - Bearer: []
    parameters:
      - name: document_id
        in: path
        type: string
        required: true
    responses:
      200:
        description: Document details fetched
    """
    request_id = str(uuid.uuid4())
    action_log_id = str(uuid.uuid4())

    try:
        current_user_id = get_jwt_identity()

        # Find document
        document = CaseDocument.objects(
            document_id=document_id,
            user_id=current_user_id,
            is_deleted=False
        ).first()

        if not document:
            return error_response(
                message="Document not found",
                error_code="DOCUMENT_NOT_FOUND",
                error_details="No document found with this ID",
                status_code=404,
                request_id=request_id,
                action_log_id=action_log_id
            )

        # Get case name
        case = Case.objects(case_id=document.case_id).first()
        case_name = case.name if case else "Unknown"

        response_data = {
            "document_id": document.document_id,
            "file_name": document.original_file_name,
            "document_type": document.document_type,
            "file_size": document.file_size,
            "case_id": document.case_id,
            "case_name": case_name,
            "uploaded_at": document.uploaded_at.isoformat(),
            "processing_status": document.processing_status,
            "ai_response_id": document.ai_response_id
        }

        return success_response(
            message="Document details fetched",
            data=response_data,
            status_code=200,
            request_id=request_id,
            action_log_id=action_log_id
        )

    except Exception as e:
        return error_response(
            message="Internal server error",
            error_code="INTERNAL_ERROR",
            error_details=str(e),
            status_code=500,
            request_id=request_id,
            action_log_id=action_log_id
        )


@api_v1_bp.route('/delete_document_by_case_id/<document_id>', methods=['DELETE'])
@jwt_required()
def delete_document(document_id):
    """
    Delete Document By ID
    ---
    tags:
      - Document Management
    security:
      - Bearer: []
    parameters:
      - name: document_id
        in: path
        type: string
        required: true
    responses:
      200:
        description: Document deleted successfully
    """
    request_id = str(uuid.uuid4())
    action_log_id = str(uuid.uuid4())

    try:
        current_user_id = get_jwt_identity()

        # Find document
        document = CaseDocument.objects(
            document_id=document_id,
            user_id=current_user_id,
            is_deleted=False
        ).first()

        if not document:
            return error_response(
                message="Document not found",
                error_code="DOCUMENT_NOT_FOUND",
                error_details="No document found with this ID",
                status_code=404,
                request_id=request_id,
                action_log_id=action_log_id
            )

        # Soft delete document
        document.soft_delete(current_user_id)

        # Update case document count
        case = Case.objects(case_id=document.case_id).first()
        if case and case.documents_count > 0:
            case.documents_count -= 1
            case.save()

        response_data = {
            "document_id": document.document_id,
            "deleted_at": document.deleted_at.isoformat()
        }

        return success_response(
            message="Document deleted successfully",
            data=response_data,
            status_code=200,
            request_id=request_id,
            action_log_id=action_log_id
        )

    except Exception as e:
        return error_response(
            message="Internal server error",
            error_code="INTERNAL_ERROR",
            error_details=str(e),
            status_code=500,
            request_id=request_id,
            action_log_id=action_log_id
        )


# ============================================================================
# NOTIFICATION ENDPOINTS
# ============================================================================

@api_v1_bp.route('/notifications', methods=['GET'])
@jwt_required()
def get_notifications():
    """
    Get User Notifications
    ---
    tags:
      - Notifications
    security:
      - Bearer: []
    parameters:
      - name: page
        in: query
        type: integer
        default: 1
        description: Page number
      - name: limit
        in: query
        type: integer
        default: 20
        description: Items per page
      - name: status
        in: query
        type: string
        enum: [unread, read, all]
        default: all
        description: Filter by notification status
      - name: type
        in: query
        type: string
        enum: [success, info, warning, error]
        description: Filter by notification type
    responses:
      200:
        description: Notifications fetched successfully
        schema:
          type: object
          properties:
            status:
              type: string
              example: success
            status_code:
              type: integer
              example: 200
            message:
              type: string
              example: Notifications fetched
            data:
              type: object
              properties:
                notifications:
                  type: array
                  items:
                    type: object
                pagination:
                  type: object
                unread_count:
                  type: integer
      401:
        description: Unauthorized
    """
    request_id = str(uuid.uuid4())
    action_log_id = str(uuid.uuid4())

    try:
        current_user_id = get_jwt_identity()
        
        # Import Notification model
        from app.models.notification import Notification

        # Get query parameters
        page = int(request.args.get('page', 1))
        limit = int(request.args.get('limit', 20))
        status_filter = request.args.get('status', 'all')
        type_filter = request.args.get('type')

        # Get headers
        ip_address = request.headers.get('X-Real-IP', request.remote_addr)
        user_agent = request.headers.get('User-Agent', '')

        # Build query
        query = {'user_id': current_user_id}
        
        if status_filter and status_filter != 'all':
            query['status'] = status_filter
        
        if type_filter:
            query['type'] = type_filter

        # Get total count
        total_items = Notification.objects(**query).count()
        total_pages = (total_items + limit - 1) // limit if limit > 0 else 0

        # Get paginated notifications
        skip = (page - 1) * limit
        notifications = Notification.objects(**query).order_by('-created_at').skip(skip).limit(limit)

        # Get unread count
        unread_count = Notification.get_unread_count(current_user_id)

        # Format response
        notifications_data = []
        for notif in notifications:
            notifications_data.append({
                "notification_id": notif.notification_id,
                "title": notif.title,
                "message": notif.message,
                "type": notif.type,
                "status": notif.status,
                "priority": notif.priority,
                "entity_type": notif.entity_type,
                "entity_id": notif.entity_id,
                "action_url": notif.action_url,
                "created_at": notif.created_at.isoformat() if notif.created_at else None,
                "read_at": notif.read_at.isoformat() if notif.read_at else None
            })

        response_data = {
            "notifications": notifications_data,
            "pagination": {
                "page": page,
                "limit": limit,
                "total_pages": total_pages,
                "total": total_items
            },
            "unread_count": unread_count
        }

        # Log action
        ActionLog(
            action_log_id=action_log_id,
            user_id=current_user_id,
            action_type='get_notifications',
            ip_address=ip_address,
            user_agent=user_agent,
            response_status='success'
        ).save()

        return success_response(
            message="Notifications fetched",
            data=response_data,
            status_code=200,
            request_id=request_id,
            action_log_id=action_log_id
        )

    except Exception as e:
        return error_response(
            message="Internal server error",
            error_code="INTERNAL_ERROR",
            error_details=str(e),
            status_code=500,
            request_id=request_id,
            action_log_id=action_log_id
        )


@api_v1_bp.route('/notifications/<notification_id>/read', methods=['PUT'])
@jwt_required()
def mark_notification_as_read(notification_id):
    """
    Mark Notification as Read
    ---
    tags:
      - Notifications
    security:
      - Bearer: []
    parameters:
      - name: notification_id
        in: path
        type: string
        required: true
        description: Notification ID to mark as read
    responses:
      200:
        description: Notification marked as read
        schema:
          type: object
          properties:
            status:
              type: string
              example: success
            status_code:
              type: integer
              example: 200
            message:
              type: string
              example: Notification marked as read
            data:
              type: object
              properties:
                notification_id:
                  type: string
                status:
                  type: string
                  example: read
                read_at:
                  type: string
                  format: date-time
      404:
        description: Notification not found
    """
    request_id = str(uuid.uuid4())
    action_log_id = str(uuid.uuid4())

    try:
        current_user_id = get_jwt_identity()
        
        # Import Notification model
        from app.models.notification import Notification

        # Get headers
        ip_address = request.headers.get('X-Real-IP', request.remote_addr)
        user_agent = request.headers.get('User-Agent', '')

        # Find notification
        notification = Notification.objects(
            notification_id=notification_id,
            user_id=current_user_id
        ).first()

        if not notification:
            ActionLog(
                action_log_id=action_log_id,
                user_id=current_user_id,
                action_type='mark_notification_read',
                ip_address=ip_address,
                user_agent=user_agent,
                request_data={"notification_id": notification_id},
                response_status='failed',
                error_details={'reason': 'Notification not found'}
            ).save()

            return error_response(
                message="Notification not found",
                error_code="NOTIFICATION_NOT_FOUND",
                error_details="No notification found with this ID",
                status_code=404,
                request_id=request_id,
                action_log_id=action_log_id
            )

        # Mark as read
        notification.mark_as_read()

        # Log action
        ActionLog(
            action_log_id=action_log_id,
            user_id=current_user_id,
            action_type='mark_notification_read',
            entity_type='notification',
            entity_id=notification.notification_id,
            ip_address=ip_address,
            user_agent=user_agent,
            response_status='success'
        ).save()

        response_data = {
            "notification_id": notification.notification_id,
            "status": notification.status,
            "read_at": notification.read_at.isoformat() if notification.read_at else None
        }

        return success_response(
            message="Notification marked as read",
            data=response_data,
            status_code=200,
            request_id=request_id,
            action_log_id=action_log_id
        )

    except Exception as e:
        return error_response(
            message="Internal server error",
            error_code="INTERNAL_ERROR",
            error_details=str(e),
            status_code=500,
            request_id=request_id,
            action_log_id=action_log_id
        )


@api_v1_bp.route('/notifications/mark-all-read', methods=['PUT'])
@jwt_required()
def mark_all_notifications_as_read():
    """
    Mark All Notifications as Read
    ---
    tags:
      - Notifications
    security:
      - Bearer: []
    responses:
      200:
        description: All notifications marked as read
        schema:
          type: object
          properties:
            status:
              type: string
              example: success
            status_code:
              type: integer
              example: 200
            message:
              type: string
              example: All notifications marked as read
            data:
              type: object
              properties:
                marked_count:
                  type: integer
                  example: 15
                marked_at:
                  type: string
                  format: date-time
      401:
        description: Unauthorized
    """
    request_id = str(uuid.uuid4())
    action_log_id = str(uuid.uuid4())

    try:
        current_user_id = get_jwt_identity()
        
        # Import Notification model
        from app.models.notification import Notification

        # Get headers
        ip_address = request.headers.get('X-Real-IP', request.remote_addr)
        user_agent = request.headers.get('User-Agent', '')

        # Mark all as read
        marked_count = Notification.mark_all_as_read(current_user_id)
        marked_at = datetime.utcnow()

        # Log action
        ActionLog(
            action_log_id=action_log_id,
            user_id=current_user_id,
            action_type='mark_all_notifications_read',
            ip_address=ip_address,
            user_agent=user_agent,
            request_data={"marked_count": marked_count},
            response_status='success'
        ).save()

        response_data = {
            "marked_count": marked_count,
            "marked_at": marked_at.isoformat()
        }

        return success_response(
            message="All notifications marked as read",
            data=response_data,
            status_code=200,
            request_id=request_id,
            action_log_id=action_log_id
        )

    except Exception as e:
        return error_response(
            message="Internal server error",
            error_code="INTERNAL_ERROR",
            error_details=str(e),
            status_code=500,
            request_id=request_id,
            action_log_id=action_log_id
        )


# ============================================================================
# ANNOUNCEMENT ENDPOINTS
# ============================================================================

@api_v1_bp.route('/get_all_announcements', methods=['GET'])
@jwt_required()
def get_all_announcements():
    """
    Get All Announcements
    ---
    tags:
      - Announcements
    security:
      - Bearer: []
    parameters:
      - name: Authorization
        in: header
        type: string
        required: true
      - name: page
        in: query
        type: integer
        default: 1
      - name: limit
        in: query
        type: integer
        default: 50
      - name: status
        in: query
        type: string
        enum: [active, archived]
        default: active
    responses:
      200:
        description: Announcements fetched successfully
    """
    request_id = str(uuid.uuid4())
    action_log_id = str(uuid.uuid4())

    try:
        from app.models.announcement import Announcement

        user_id = get_jwt_identity()

        # Get query parameters
        page = int(request.args.get('page', 1))
        limit = int(request.args.get('limit', 50))
        status_filter = request.args.get('status', 'active').strip()

        # Build query
        query = {'is_deleted': False}
        if status_filter in ['active', 'archived']:
            query['status'] = status_filter

        # Pagination
        skip = (page - 1) * limit
        total_count = Announcement.objects(__raw__=query).count()

        # Fetch announcements
        announcements = Announcement.objects(__raw__=query).skip(skip).limit(limit).order_by('-created_at')

        # Prepare response data
        announcements_data = []
        for announcement in announcements:
            announcements_data.append({
                'announcement_id': announcement.announcement_id,
                'title': announcement.title,
                'message': announcement.message,
                'created_at': announcement.created_at.isoformat() + 'Z' if announcement.created_at else None,
                'updated_at': announcement.updated_at.isoformat() + 'Z' if announcement.updated_at else None,
                'status': announcement.status,
                'created_by': announcement.created_by_name if announcement.created_by_name else 'Admin'
            })

        # Log success
        ActionLog(
            action_log_id=action_log_id,
            user_id=user_id,
            action_type='get_all_announcements',
            ip_address=request.headers.get('X-Real-IP', request.remote_addr),
            user_agent=request.headers.get('User-Agent', ''),
            request_data={'page': page, 'limit': limit, 'status': status_filter},
            response_status='success'
        ).save()

        return success_response(
            message="Announcements fetched successfully",
            data={
                'announcements': announcements_data,
                'pagination': {
                    'page': page,
                    'limit': limit,
                    'total': total_count
                }
            },
            status_code=200,
            request_id=request_id,
            action_log_id=action_log_id
        )

    except Exception as e:
        ActionLog(
            action_log_id=action_log_id,
            user_id=get_jwt_identity() if 'user_id' not in locals() else user_id,
            action_type='get_all_announcements',
            ip_address=request.headers.get('X-Real-IP', request.remote_addr),
            user_agent=request.headers.get('User-Agent', ''),
            response_status='error',
            error_details={'exception': str(e)}
        ).save()

        return error_response(
            message="Internal server error",
            error_code="INTERNAL_ERROR",
            error_details=str(e),
            status_code=500,
            request_id=request_id,
            action_log_id=action_log_id
        )


@api_v1_bp.route('/get_announcement_by_id/<announcement_id>', methods=['GET'])
@jwt_required()
def get_announcement_by_id(announcement_id):
    """
    Get Announcement By ID
    ---
    tags:
      - Announcements
    security:
      - Bearer: []
    parameters:
      - name: Authorization
        in: header
        type: string
        required: true
      - name: announcement_id
        in: path
        type: string
        required: true
    responses:
      200:
        description: Announcement details fetched
      404:
        description: Announcement not found
    """
    request_id = str(uuid.uuid4())
    action_log_id = str(uuid.uuid4())

    try:
        from app.models.announcement import Announcement

        user_id = get_jwt_identity()

        # Find announcement
        announcement = Announcement.objects(announcement_id=announcement_id, is_deleted=False).first()
        if not announcement:
            ActionLog(
                action_log_id=action_log_id,
                user_id=user_id,
                action_type='get_announcement_by_id',
                ip_address=request.headers.get('X-Real-IP', request.remote_addr),
                user_agent=request.headers.get('User-Agent', ''),
                request_data={'announcement_id': announcement_id},
                response_status='failed',
                error_details={'reason': 'Announcement not found'}
            ).save()

            return error_response(
                message="Announcement not found",
                error_code="ANNOUNCEMENT_NOT_FOUND",
                error_details="No announcement found with this ID",
                status_code=404,
                request_id=request_id,
                action_log_id=action_log_id
            )

        # Increment view count
        announcement.increment_view_count()

        # Log success
        ActionLog(
            action_log_id=action_log_id,
            user_id=user_id,
            action_type='get_announcement_by_id',
            entity_type='announcement',
            entity_id=announcement.announcement_id,
            ip_address=request.headers.get('X-Real-IP', request.remote_addr),
            user_agent=request.headers.get('User-Agent', ''),
            request_data={'announcement_id': announcement_id},
            response_status='success'
        ).save()

        return success_response(
            message="Announcement details fetched",
            data={
                'announcement_id': announcement.announcement_id,
                'title': announcement.title,
                'message': announcement.message,
                'created_at': announcement.created_at.isoformat() + 'Z' if announcement.created_at else None,
                'updated_at': announcement.updated_at.isoformat() + 'Z' if announcement.updated_at else None,
                'status': announcement.status,
                'created_by': announcement.created_by_name if announcement.created_by_name else 'Admin',
                'view_count': announcement.view_count
            },
            status_code=200,
            request_id=request_id,
            action_log_id=action_log_id
        )

    except Exception as e:
        ActionLog(
            action_log_id=action_log_id,
            user_id=get_jwt_identity() if 'user_id' not in locals() else user_id,
            action_type='get_announcement_by_id',
            ip_address=request.headers.get('X-Real-IP', request.remote_addr),
            user_agent=request.headers.get('User-Agent', ''),
            request_data={'announcement_id': announcement_id},
            response_status='error',
            error_details={'exception': str(e)}
        ).save()

        return error_response(
            message="Internal server error",
            error_code="INTERNAL_ERROR",
            error_details=str(e),
            status_code=500,
            request_id=request_id,
            action_log_id=action_log_id
        )


# ============================================================================
# CLIENT MANAGEMENT ENDPOINTS (USER)
# ============================================================================

@api_v1_bp.route('/user/add_client', methods=['POST'])
@jwt_required()
def user_add_client():
    """
    Add New Client (User)
    ---
    tags:
      - User - Clients
    security:
      - Bearer: []
    parameters:
      - name: Authorization
        in: header
        type: string
        required: true
        description: Bearer token (JWT)
      - name: body
        in: body
        required: true
        schema:
          type: object
          required:
            - client_name
          properties:
            client_name:
              type: string
              example: "Stockholm"
            email:
              type: string
              example: "contact@stockholm.com"
            phone:
              type: string
              example: "+46 8 123 4567"
            address:
              type: string
              example: "Stockholm, Sweden"
            contact_person:
              type: string
              example: "John Doe"
            description:
              type: string
              example: "Corporate client"
    responses:
      201:
        description: Client added successfully
      400:
        description: Validation error
      401:
        description: Unauthorized
    """
    request_id = str(uuid.uuid4())
    action_log_id = str(uuid.uuid4())

    try:
        from app.models.client import Client

        user_id = get_jwt_identity()
        data = request.get_json()

        # Validate
        client_name = data.get('client_name', '').strip()
        if not client_name:
            return error_response(
                message="Client name is required",
                error_code="VALIDATION_ERROR",
                error_details="client_name is required",
                status_code=400,
                request_id=request_id,
                action_log_id=action_log_id
            )

        # Check if client already exists
        existing_client = Client.objects(client_name=client_name, is_deleted=False).first()
        if existing_client:
            return error_response(
                message="Client already exists",
                error_code="VALIDATION_ERROR",
                error_details="Client with this name already exists",
                status_code=400,
                request_id=request_id,
                action_log_id=action_log_id
            )

        # Create client
        new_client = Client(
            client_name=client_name,
            email=data.get('email', '').strip(),
            phone=data.get('phone', '').strip(),
            address=data.get('address', '').strip(),
            contact_person=data.get('contact_person', '').strip(),
            description=data.get('description', '').strip(),
            created_by=user_id,
            created_at=datetime.utcnow(),
            updated_at=datetime.utcnow()
        )
        new_client.save()

        # Log success
        ActionLog(
            action_log_id=action_log_id,
            user_id=user_id,
            action_type='user_add_client',
            entity_type='client',
            entity_id=new_client.client_id,
            ip_address=request.headers.get('X-Real-IP', request.remote_addr),
            user_agent=request.headers.get('User-Agent', ''),
            request_data=data,
            response_status='success'
        ).save()

        return success_response(
            message="Client added successfully",
            data={
                'client_id': new_client.client_id,
                'client_name': new_client.client_name,
                'created_at': new_client.created_at.isoformat() + 'Z'
            },
            status_code=201,
            request_id=request_id,
            action_log_id=action_log_id
        )

    except Exception as e:
        ActionLog(
            action_log_id=action_log_id,
            user_id=get_jwt_identity() if 'user_id' not in locals() else user_id,
            action_type='user_add_client',
            ip_address=request.headers.get('X-Real-IP', request.remote_addr),
            user_agent=request.headers.get('User-Agent', ''),
            request_data=data if 'data' in locals() else {},
            response_status='error',
            error_details={'exception': str(e)}
        ).save()

        return error_response(
            message="Internal server error",
            error_code="INTERNAL_ERROR",
            error_details=str(e),
            status_code=500,
            request_id=request_id,
            action_log_id=action_log_id
        )


@api_v1_bp.route('/user/get_all_clients', methods=['GET'])
@jwt_required()
def user_get_all_clients():
    """
    Get All Clients (User)
    ---
    tags:
      - User - Clients
    security:
      - Bearer: []
    parameters:
      - name: Authorization
        in: header
        type: string
        required: true
      - name: page
        in: query
        type: integer
        default: 1
      - name: limit
        in: query
        type: integer
        default: 50
    responses:
      200:
        description: Clients fetched successfully
      401:
        description: Unauthorized
    """
    request_id = str(uuid.uuid4())
    action_log_id = str(uuid.uuid4())

    try:
        from app.models.client import Client
        from app.models.case import Case

        user_id = get_jwt_identity()

        # Get query parameters
        page = int(request.args.get('page', 1))
        limit = int(request.args.get('limit', 50))

        # Pagination
        skip = (page - 1) * limit
        total_count = Client.objects(is_deleted=False).count()

        # Fetch clients
        clients = Client.objects(is_deleted=False).skip(skip).limit(limit).order_by('-created_at')

        # Prepare response data
        clients_data = []
        for client in clients:
            # Update case counts
            client.update_case_counts()

            clients_data.append({
                'client_id': client.client_id,
                'client_name': client.client_name,
                'active_cases': client.active_cases_count,
                'inactive_cases': client.inactive_cases_count,
                'total_cases': client.total_cases_count
            })

        # Log success
        ActionLog(
            action_log_id=action_log_id,
            user_id=user_id,
            action_type='user_get_all_clients',
            ip_address=request.headers.get('X-Real-IP', request.remote_addr),
            user_agent=request.headers.get('User-Agent', ''),
            request_data={'page': page, 'limit': limit},
            response_status='success'
        ).save()

        return success_response(
            message="Clients fetched successfully",
            data={
                'clients': clients_data,
                'pagination': {
                    'page': page,
                    'limit': limit,
                    'total': total_count
                }
            },
            status_code=200,
            request_id=request_id,
            action_log_id=action_log_id
        )

    except Exception as e:
        ActionLog(
            action_log_id=action_log_id,
            user_id=get_jwt_identity() if 'user_id' not in locals() else user_id,
            action_type='user_get_all_clients',
            ip_address=request.headers.get('X-Real-IP', request.remote_addr),
            user_agent=request.headers.get('User-Agent', ''),
            response_status='error',
            error_details={'exception': str(e)}
        ).save()

        return error_response(
            message="Internal server error",
            error_code="INTERNAL_ERROR",
            error_details=str(e),
            status_code=500,
            request_id=request_id,
            action_log_id=action_log_id
        )


@api_v1_bp.route('/user/get_client_by_id/<client_id>', methods=['GET'])
@jwt_required()
def user_get_client_by_id(client_id):
    """
    Get Client By ID (User)
    ---
    tags:
      - User - Clients
    security:
      - Bearer: []
    parameters:
      - name: Authorization
        in: header
        type: string
        required: true
      - name: client_id
        in: path
        type: string
        required: true
    responses:
      200:
        description: Client details fetched
      401:
        description: Unauthorized
      404:
        description: Client not found
    """
    request_id = str(uuid.uuid4())
    action_log_id = str(uuid.uuid4())

    try:
        from app.models.client import Client
        from app.models.case import Case

        user_id = get_jwt_identity()

        # Find client
        client = Client.objects(client_id=client_id, is_deleted=False).first()
        if not client:
            ActionLog(
                action_log_id=action_log_id,
                user_id=user_id,
                action_type='user_get_client_by_id',
                ip_address=request.headers.get('X-Real-IP', request.remote_addr),
                user_agent=request.headers.get('User-Agent', ''),
                request_data={'client_id': client_id},
                response_status='failed',
                error_details={'reason': 'Client not found'}
            ).save()

            return error_response(
                message="Client not found",
                error_code="CLIENT_NOT_FOUND",
                error_details="No client found with this ID",
                status_code=404,
                request_id=request_id,
                action_log_id=action_log_id
            )

        # Get cases for this client
        cases = Case.objects(client_id=client_id, is_deleted=False)
        cases_data = [{
            'case_id': case.case_id,
            'name': case.name,
            'case_number': case.case_number,
            'status': case.status,
            'start_date': case.start_date.isoformat() + 'Z' if case.start_date else None,
            'end_date': case.end_date.isoformat() + 'Z' if case.end_date else None,
            'documents_count': case.documents_count
        } for case in cases]

        # Log success
        ActionLog(
            action_log_id=action_log_id,
            user_id=user_id,
            action_type='user_get_client_by_id',
            entity_type='client',
            entity_id=client.client_id,
            ip_address=request.headers.get('X-Real-IP', request.remote_addr),
            user_agent=request.headers.get('User-Agent', ''),
            request_data={'client_id': client_id},
            response_status='success'
        ).save()

        return success_response(
            message="Client details fetched",
            data={
                'client_id': client.client_id,
                'client_name': client.client_name,
                'cases': cases_data
            },
            status_code=200,
            request_id=request_id,
            action_log_id=action_log_id
        )

    except Exception as e:
        ActionLog(
            action_log_id=action_log_id,
            user_id=get_jwt_identity() if 'user_id' not in locals() else user_id,
            action_type='user_get_client_by_id',
            ip_address=request.headers.get('X-Real-IP', request.remote_addr),
            user_agent=request.headers.get('User-Agent', ''),
            request_data={'client_id': client_id},
            response_status='error',
            error_details={'exception': str(e)}
        ).save()

        return error_response(
            message="Internal server error",
            error_code="INTERNAL_ERROR",
            error_details=str(e),
            status_code=500,
            request_id=request_id,
            action_log_id=action_log_id
        )


@api_v1_bp.route('/update/announcements/<announcement_id>', methods=['PUT'])
@jwt_required()
def update_announcement(announcement_id):
    """
    Update Announcement
    ---
    tags:
      - Announcements
    security:
      - Bearer: []
    parameters:
      - name: Authorization
        in: header
        type: string
        required: true
      - name: announcement_id
        in: path
        type: string
        required: true
      - name: body
        in: body
        required: true
        schema:
          type: object
          properties:
            title:
              type: string
              example: "Updated: Maintenance Rescheduled"
            message:
              type: string
              example: "Maintenance has been moved to June 20..."
            status:
              type: string
              enum: [active, archived]
    responses:
      200:
        description: Announcement updated successfully
      400:
        description: Invalid announcement ID or data
      401:
        description: Unauthorized access
      403:
        description: Permission denied
      404:
        description: Announcement not found
    """
    request_id = str(uuid.uuid4())
    action_log_id = str(uuid.uuid4())

    try:
        from app.models.announcement import Announcement

        user_id = get_jwt_identity()
        data = request.get_json()

        # Check if user is admin (only admins can update announcements)
        user = User.objects(user_id=user_id).first()
        if not user or user.user_type != 'AD':
            return error_response(
                message="Permission denied",
                error_code="FORBIDDEN",
                error_details="Admin access required to update announcements",
                status_code=403,
                request_id=request_id,
                action_log_id=action_log_id
            )

        # Find announcement
        announcement = Announcement.objects(announcement_id=announcement_id, is_deleted=False).first()
        if not announcement:
            ActionLog(
                action_log_id=action_log_id,
                user_id=user_id,
                action_type='update_announcement',
                ip_address=request.headers.get('X-Real-IP', request.remote_addr),
                user_agent=request.headers.get('User-Agent', ''),
                request_data={'announcement_id': announcement_id},
                response_status='failed',
                error_details={'reason': 'Announcement not found'}
            ).save()

            return error_response(
                message="Announcement not found",
                error_code="ANNOUNCEMENT_NOT_FOUND",
                error_details="No announcement found with this ID",
                status_code=404,
                request_id=request_id,
                action_log_id=action_log_id
            )

        # Update fields
        if 'title' in data and data['title'].strip():
            announcement.title = data['title'].strip()
        if 'message' in data and data['message'].strip():
            announcement.message = data['message'].strip()
        if 'status' in data and data['status'] in ['active', 'archived']:
            announcement.status = data['status']

        announcement.updated_at = datetime.utcnow()
        announcement.updated_by = user_id
        announcement.save()

        # Log success
        ActionLog(
            action_log_id=action_log_id,
            user_id=user_id,
            action_type='update_announcement',
            entity_type='announcement',
            entity_id=announcement.announcement_id,
            ip_address=request.headers.get('X-Real-IP', request.remote_addr),
            user_agent=request.headers.get('User-Agent', ''),
            request_data=data,
            response_status='success'
        ).save()

        return success_response(
            message="Announcement updated successfully",
            data={
                'announcement_id': announcement.announcement_id,
                'title': announcement.title,
                'message': announcement.message,
                'created_at': announcement.created_at.isoformat() + 'Z' if announcement.created_at else None,
                'updated_at': announcement.updated_at.isoformat() + 'Z' if announcement.updated_at else None,
                'status': announcement.status,
                'created_by': announcement.created_by_name if announcement.created_by_name else 'Admin'
            },
            status_code=200,
            request_id=request_id,
            action_log_id=action_log_id
        )

    except Exception as e:
        ActionLog(
            action_log_id=action_log_id,
            user_id=get_jwt_identity() if 'user_id' not in locals() else user_id,
            action_type='update_announcement',
            ip_address=request.headers.get('X-Real-IP', request.remote_addr),
            user_agent=request.headers.get('User-Agent', ''),
            request_data=data if 'data' in locals() else {},
            response_status='error',
            error_details={'exception': str(e)}
        ).save()

        return error_response(
            message="Internal server error",
            error_code="INTERNAL_ERROR",
            error_details=str(e),
            status_code=500,
            request_id=request_id,
            action_log_id=action_log_id
        )


@api_v1_bp.route('/delete/announcements/<announcement_id>', methods=['DELETE'])
@jwt_required()
def delete_announcement(announcement_id):
    """
    Delete Announcement
    ---
    tags:
      - Announcements
    security:
      - Bearer: []
    parameters:
      - name: Authorization
        in: header
        type: string
        required: true
      - name: announcement_id
        in: path
        type: string
        required: true
    responses:
      200:
        description: Announcement deleted successfully
      400:
        description: Invalid announcement ID
      401:
        description: Unauthorized access
      403:
        description: Permission denied - Admin access required
      404:
        description: Announcement not found or already deleted
    """
    request_id = str(uuid.uuid4())
    action_log_id = str(uuid.uuid4())

    try:
        from app.models.announcement import Announcement

        user_id = get_jwt_identity()

        # Check if user is admin (only admins can delete announcements)
        user = User.objects(user_id=user_id).first()
        if not user or user.user_type != 'AD':
            return error_response(
                message="Permission denied: Admin access required",
                error_code="FORBIDDEN",
                error_details="Only administrators can delete announcements",
                status_code=403,
                request_id=request_id,
                action_log_id=action_log_id
            )

        # Find announcement
        announcement = Announcement.objects(announcement_id=announcement_id, is_deleted=False).first()
        if not announcement:
            ActionLog(
                action_log_id=action_log_id,
                user_id=user_id,
                action_type='delete_announcement',
                ip_address=request.headers.get('X-Real-IP', request.remote_addr),
                user_agent=request.headers.get('User-Agent', ''),
                request_data={'announcement_id': announcement_id},
                response_status='failed',
                error_details={'reason': 'Announcement not found'}
            ).save()

            return error_response(
                message="Announcement not found or already deleted",
                error_code="ANNOUNCEMENT_NOT_FOUND",
                error_details="No announcement found with this ID",
                status_code=404,
                request_id=request_id,
                action_log_id=action_log_id
            )

        # Soft delete
        announcement.is_deleted = True
        announcement.deleted_at = datetime.utcnow()
        announcement.deleted_by = user_id
        announcement.save()

        # Log success
        ActionLog(
            action_log_id=action_log_id,
            user_id=user_id,
            action_type='delete_announcement',
            entity_type='announcement',
            entity_id=announcement.announcement_id,
            ip_address=request.headers.get('X-Real-IP', request.remote_addr),
            user_agent=request.headers.get('User-Agent', ''),
            request_data={'announcement_id': announcement_id},
            response_status='success'
        ).save()

        return success_response(
            message="Announcement deleted successfully",
            data={},
            status_code=200,
            request_id=request_id,
            action_log_id=action_log_id
        )

    except Exception as e:
        ActionLog(
            action_log_id=action_log_id,
            user_id=get_jwt_identity() if 'user_id' not in locals() else user_id,
            action_type='delete_announcement',
            ip_address=request.headers.get('X-Real-IP', request.remote_addr),
            user_agent=request.headers.get('User-Agent', ''),
            request_data={'announcement_id': announcement_id},
            response_status='error',
            error_details={'exception': str(e)}
        ).save()

        return error_response(
            message="Internal server error",
            error_code="INTERNAL_ERROR",
            error_details=str(e),
            status_code=500,
            request_id=request_id,
            action_log_id=action_log_id
        )
